// Excursion Booking Website - Version 4 (HTML/CSS/JS) - Elegant Cultural Design

const themes = {
  historical: {
    name: 'Историческая',
    color: '#6B4E3D',
    description: 'Погрузитесь в многовековую историю города',
    duration: '3 часа',
    price: 'от 1500₽',
    quote: 'История — это не просто прошлое, это живая память, которая формирует наше настоящее.',
    features: ['Эксклюзивные маршруты', 'Исторические факты', 'Архитектурные детали'],
    guide: 'Анна Петрова'
  },
  gastronomic: {
    name: 'Гастрономическая',
    color: '#6B4E3D',
    description: 'Откройте вкусы и традиции местной кухни',
    duration: '2.5 часа',
    price: 'от 2000₽',
    quote: 'Кухня — это язык культуры, который говорит через вкусы и ароматы.',
    features: ['Дегустации', 'Исторические рецепты', 'Встречи с шеф-поварами'],
    guide: 'Михаил Соколов'
  },
  nature: {
    name: 'Природная',
    color: '#6B4E3D',
    description: 'Насладитесь красотой природы в черте города',
    duration: '4 часа',
    price: 'от 1800₽',
    quote: 'Природа — это величайший учитель, который показывает нам красоту и гармонию.',
    features: ['Экологические тропы', 'Наблюдение за природой', 'Фотосессии'],
    guide: 'Елена Волкова'
  },
  architectural: {
    name: 'Архитектурная',
    color: '#6B4E3D',
    description: 'Изучите уникальную архитектуру разных эпох',
    duration: '3.5 часа',
    price: 'от 1700₽',
    quote: 'Архитектура — это застывшая музыка, которая рассказывает истории эпох.',
    features: ['Архитектурные стили', 'Секреты строительства', 'Современные проекты'],
    guide: 'Дмитрий Новиков'
  }
}

let currentSlide = 0
let slides = []
let dots = []
let slideInterval
let selectedTheme = ''
let selectedDate = ''

function initPage() {
  const currentPage = window.location.pathname.includes('book.html') ? 'book' : 'index'
  
  if (currentPage === 'book') {
    initBookPage()
  } else {
    initIndexPage()
  }
  
  initSlider()
}

function initIndexPage() {
  loadTours()
  updateStats()
  
  setInterval(() => {
    loadTours()
    updateStats()
  }, 500)
}

function loadTours() {
  const saved = localStorage.getItem('excursionsTours')
  let tours = []
  
  if (saved) {
    try {
      tours = JSON.parse(saved)
    } catch (e) {
      tours = []
    }
  } else {
    tours = [
      {
        id: 1,
        route: 'Тайны Красной площади',
        guide: 'Анна Петрова',
        date: '2025-04-15',
        rating: 5,
        duration: '3 часа',
        theme: 'Историческая',
        highlights: ['Кремль', 'Собор Василия Блаженного', 'ГУМ']
      },
      {
        id: 2,
        route: 'Гастрономические секреты Замоскворечья',
        guide: 'Михаил Соколов',
        date: '2025-04-12',
        rating: 4,
        duration: '2.5 часа',
        theme: 'Гастрономическая',
        highlights: ['Традиционные блюда', 'Исторические кафе', 'Дегустации']
      },
      {
        id: 3,
        route: 'Природные сокровища Коломенского',
        guide: 'Елена Волкова',
        date: '2025-04-10',
        rating: 5,
        duration: '4 часа',
        theme: 'Природная',
        highlights: ['Древние дубы', 'Яблоневые сады', 'Речные пейзажи']
      }
    ]
    localStorage.setItem('excursionsTours', JSON.stringify(tours))
  }
  
  renderTours(tours)
}

function renderTours(tours) {
  const container = document.getElementById('tours-container')
  if (!container) return
  
  if (tours.length === 0) {
    container.innerHTML = `
      <div class="empty-elegant">
        <div class="empty-icon-elegant">
          <img src="icons/scroll-text.svg" alt="ScrollText" width="48" height="48">
        </div>
        <h3 class="empty-title-elegant">Начните свое путешествие</h3>
        <p class="empty-text-elegant">Запишитесь на первую экскурсию и откройте для себя культурное наследие</p>
      </div>
    `
    return
  }
  
  container.innerHTML = tours.map(tour => {
    const date = new Date(tour.date)
    const formattedDate = date.toLocaleDateString('ru-RU', { day: 'numeric', month: 'long', year: 'numeric' })
    
    return `
      <div class="tour-timeline-item">
        <div class="timeline-marker">
          <div class="marker-dot"></div>
          <div class="marker-line"></div>
        </div>
        <div class="tour-timeline-content">
          <div class="tour-timeline-header">
            <div class="tour-theme-elegant">${tour.theme}</div>
            <div class="tour-rating-elegant">
              ${Array.from({ length: 5 }).map((_, i) => 
                `<img src="icons/star.svg" alt="Star" width="14" height="14" style="filter: ${i < tour.rating ? 'none' : 'grayscale(100%) opacity(0.3)'}">`
              ).join('')}
            </div>
          </div>
          <h3 class="tour-route-elegant">${tour.route}</h3>
          <div class="tour-details-elegant">
            <div class="tour-detail-elegant">
              <img src="icons/user.svg" alt="User" width="14" height="14">
              <span>${tour.guide}</span>
            </div>
            <div class="tour-detail-elegant">
              <img src="icons/calendar.svg" alt="Calendar" width="14" height="14">
              <span>${formattedDate}</span>
            </div>
            <div class="tour-detail-elegant">
              <img src="icons/clock.svg" alt="Clock" width="14" height="14">
              <span>${tour.duration}</span>
            </div>
          </div>
          ${tour.highlights ? `
            <div class="tour-highlights-elegant">
              ${tour.highlights.map(h => `<span class="highlight-elegant">${h}</span>`).join('')}
            </div>
          ` : ''}
        </div>
      </div>
    `
  }).join('')
}

function updateStats() {
  const saved = localStorage.getItem('excursionsTours')
  let tours = []
  
  if (saved) {
    try {
      tours = JSON.parse(saved)
    } catch (e) {
      tours = []
    }
  }
  
  const totalTours = tours.length
  const averageRating = tours.length > 0
    ? (tours.reduce((sum, t) => sum + t.rating, 0) / tours.length).toFixed(1)
    : 0
  const upcomingTours = tours.filter(t => new Date(t.date) >= new Date()).length
  const completedTours = tours.filter(t => t.rating === 5).length
  
  const totalEl = document.getElementById('total-tours')
  const avgEl = document.getElementById('avg-rating')
  const upcomingEl = document.getElementById('upcoming-tours')
  const completedEl = document.getElementById('completed-tours')
  
  if (totalEl) totalEl.textContent = totalTours
  if (avgEl) avgEl.textContent = averageRating
  if (upcomingEl) upcomingEl.textContent = upcomingTours
  if (completedEl) completedEl.textContent = completedTours
}

function initSlider() {
  slides = document.querySelectorAll('.slide')
  dots = document.querySelectorAll('.dot')
  
  if (slides.length === 0) return
  
  const prevBtn = document.getElementById('slider-prev')
  const nextBtn = document.getElementById('slider-next')
  
  if (prevBtn) {
    prevBtn.addEventListener('click', () => {
      currentSlide = (currentSlide - 1 + slides.length) % slides.length
      updateSlider()
    })
  }
  
  if (nextBtn) {
    nextBtn.addEventListener('click', () => {
      currentSlide = (currentSlide + 1) % slides.length
      updateSlider()
    })
  }
  
  dots.forEach((dot, index) => {
    dot.addEventListener('click', () => {
      currentSlide = index
      updateSlider()
    })
  })
  
  updateSlider()
  
  slideInterval = setInterval(() => {
    currentSlide = (currentSlide + 1) % slides.length
    updateSlider()
  }, 3000)
}

function updateSlider() {
  slides.forEach((slide, index) => {
    if (index === currentSlide) {
      slide.classList.add('active')
    } else {
      slide.classList.remove('active')
    }
  })
  
  dots.forEach((dot, index) => {
    if (index === currentSlide) {
      dot.classList.add('active')
    } else {
      dot.classList.remove('active')
    }
  })
}

function initBookPage() {
  renderThemes()
  
  const dateInput = document.getElementById('date-input')
  if (dateInput) {
    const today = new Date()
    const maxDate = new Date(today.getTime() + 90 * 24 * 60 * 60 * 1000)
    dateInput.min = today.toISOString().split('T')[0]
    dateInput.max = maxDate.toISOString().split('T')[0]
    
    dateInput.addEventListener('change', (e) => {
      selectedDate = e.target.value
      updateFormProgress()
      toggleDateSection(true)
    })
  }
  
  const form = document.getElementById('book-form')
  if (form) {
    form.addEventListener('submit', handleSubmit)
  }
  
  updateFormProgress()
}

function renderThemes() {
  const container = document.getElementById('themes-container')
  if (!container) return
  
  container.innerHTML = Object.entries(themes).map(([id, theme]) => {
    const isSelected = selectedTheme === id
    return `
      <button type="button" class="theme-elegant-card ${isSelected ? 'selected' : ''}" 
              data-theme="${id}">
        <div class="theme-elegant-icon" style="border-color: ${isSelected ? theme.color : 'var(--border)'}; background: ${isSelected ? 'rgba(107, 78, 61, 0.08)' : 'transparent'}">
          <img src="icons/${id === 'historical' ? 'building2' : id === 'gastronomic' ? 'utensils-crossed' : id === 'nature' ? 'tree-pine' : 'camera'}.svg" 
               alt="${theme.name}" width="40" height="40" style="color: ${theme.color}">
        </div>
        <div class="theme-elegant-content">
          <h3 class="theme-elegant-name">${theme.name}</h3>
          <p class="theme-elegant-description">${theme.description}</p>
          <div class="theme-elegant-quote">
            <img src="icons/quote.svg" alt="Quote" width="16" height="16">
            <span>${theme.quote}</span>
          </div>
          <div class="theme-elegant-features">
            ${theme.features.map(f => `
              <span class="feature-elegant-tag">${f}</span>
            `).join('')}
          </div>
          <div class="theme-elegant-meta">
            <div class="meta-elegant-item">
              <img src="icons/clock.svg" alt="Clock" width="14" height="14">
              <span>${theme.duration}</span>
            </div>
            <div class="meta-elegant-item">
              <span>${theme.price}</span>
            </div>
          </div>
        </div>
        ${isSelected ? `
          <div class="theme-check-elegant">
            <img src="icons/check-circle.svg" alt="Check" width="28" height="28">
          </div>
        ` : ''}
      </button>
    `
  }).join('')
  
  container.querySelectorAll('.theme-elegant-card').forEach(card => {
    card.addEventListener('click', () => {
      selectedTheme = card.dataset.theme
      renderThemes()
      updateFormProgress()
      toggleDateSection(true)
    })
  })
}

function toggleDateSection(show) {
  const section = document.getElementById('date-section')
  if (section) {
    if (show && selectedTheme) {
      section.style.display = 'block'
      setTimeout(() => {
        section.style.opacity = '1'
        section.style.height = 'auto'
      }, 10)
    } else {
      section.style.opacity = '0'
      section.style.height = '0'
      setTimeout(() => {
        section.style.display = 'none'
      }, 300)
    }
  }
}

function updateFormProgress() {
  let progress = 0
  if (selectedTheme) progress += 50
  if (selectedDate) progress += 50
  
  const progressBar = document.getElementById('progress-fill')
  if (progressBar) {
    progressBar.style.width = `${progress}%`
  }
  
  const progressText = document.getElementById('progress-text')
  if (progressText) {
    progressText.textContent = `${progress}%`
  }
  
  const submitBtn = document.getElementById('submit-btn')
  if (submitBtn) {
    submitBtn.disabled = !selectedTheme || !selectedDate
  }
}

function handleSubmit(e) {
  e.preventDefault()
  
  if (!selectedTheme || !selectedDate) return
  
  const saved = localStorage.getItem('excursionsTours')
  const tours = saved ? JSON.parse(saved) : []
  
  const themeData = themes[selectedTheme]
  
  const newTour = {
    id: Date.now(),
    route: themeData.name,
    guide: themeData.guide,
    date: selectedDate,
    rating: 0,
    duration: themeData.duration,
    theme: themeData.name,
    highlights: themeData.features
  }
  
  tours.push(newTour)
  localStorage.setItem('excursionsTours', JSON.stringify(tours))
  
  const successMsg = document.getElementById('success-message')
  if (successMsg) {
    successMsg.style.display = 'flex'
  }
  
  setTimeout(() => {
    window.location.href = 'index.html'
  }, 2000)
}

// Initialize on page load
document.addEventListener('DOMContentLoaded', initPage)

